function segmentedImage = segmentImageLFDA(imgPath, fgSample, bgSample)
%SEGMENTIMAGELDALFDA Segment image using 1-pass LFDA (O-MMFT)
%
%   Input:
%       imgPath : path to the input image
%       fgSample: pixels of foreground scribbles
%       bgSample: pixels of background scribbles
% 
%   Output:
%       segmentedImage: binary image representing the output segmentation
%
% (c) Moustafa Meshry, moustafa.meshry@alexu.edu.eg
%     Department of Compter and Systems Engineering, Alexandria University, Egypt.

    % extract features for all pixels
    imgFeatures = extractFeatures(imgPath, fgSample, bgSample);
    [height, width, featureVectorSize] = size(imgFeatures);
    
    % Create user fg/bg scribbles binary mask
    fgScribblesMask = false(height, width);
    bgScribblesMask = false(height, width);
    fgScribblesMask(sub2ind(size(fgScribblesMask), ...
        fgSample(:, 1), fgSample(:, 2))) = true;
    bgScribblesMask(sub2ind(size(bgScribblesMask), ...
        bgSample(:, 1), bgSample(:, 2))) = true;
    
    % select features of user-labeled foreground
    fgSampleMask3D = repmat(fgScribblesMask, 1, featureVectorSize);
    fgSampleMask3D = reshape(fgSampleMask3D, size(imgFeatures));
    fgFeatures = imgFeatures(fgSampleMask3D);
    fgFeatures = reshape(fgFeatures, [], featureVectorSize);

    % select features of user-labeled background
    bgSampleMask3D = repmat(bgScribblesMask, 1, featureVectorSize);
    bgSampleMask3D = reshape(bgSampleMask3D, size(imgFeatures));
    bgFeatures = imgFeatures(bgSampleMask3D);
    bgFeatures = reshape(bgFeatures, [], featureVectorSize);

    % LDA Training
    trainSample = [fgFeatures; bgFeatures];
    trainLabels = ones(size(trainSample, 1), 1);
    trainLabels(size(fgFeatures, 1) + 1 : end) = 0;

    %LFDA
    W = LFDA(trainSample', trainLabels);
    pixelsVec = reshape(imgFeatures, height * width, []);
    pixelsScores = pixelsVec * W;

    % reduce dimensionality
    kDim = 6;
    pixelsScores(:, kDim + 1 : end) = [];

    posTrainScores = pixelsScores(fgScribblesMask(:), :);
    negTrainScores = pixelsScores(bgScribblesMask(:), :);
    [~,d1]=knnsearch(posTrainScores,pixelsScores,'k',2);
    [~,d2]=knnsearch(negTrainScores,pixelsScores,'k',2);
    ratioclass=log(sum(d1,2)./sum(d2,2));
    reshapedratioclass=reshape(ratioclass, height, width);
    bgMask=reshapedratioclass>0;

    % Color background pixels for output
    [y, x] = find(bgMask);
    segmentedImage = repmat(255, height, width);
    segmentedImage(sub2ind(size(segmentedImage), y, x)) = 0;
    segmentedImage = postProcess(segmentedImage, fgScribblesMask, ...
        bgScribblesMask);
end

